%% Defining the area of interest for our Map

%Define the range of longitude and latitude values
minLongitude = 7.6039;   % Specify the minimum longitude value
maxLongitude = 7.71; % Specify the maximum longitude value
minLatitude = 45.0386; % Specify the minimum latitude value
maxLatitude = 45.1094; % Specify the maximum latitude value

R = 6371000;

lat_min_rad = deg2rad(minLatitude);
lat_max_rad = deg2rad(maxLatitude);
lon_min_rad = deg2rad(minLongitude);
lon_max_rad = deg2rad(maxLongitude);

delta_lat = lat_max_rad - lat_min_rad;
delta_lon = lon_max_rad - lon_min_rad;

a_height = sin(delta_lat / 2)^2;
c_height = 2 * atan2(sqrt(a_height), sqrt(1 - a_height));
height_in_meters = R * c_height;

avg_lat = (lat_min_rad + lat_max_rad) / 2;
a_width = cos(avg_lat) * sin(delta_lon / 2)^2;
c_width = 2 * atan2(sqrt(a_width), sqrt(1 - a_width));
width_in_meters = R * c_width;


%% Extract buildings location from GeoJSON file from openstreetmap and convert to xy coordinates


% Specify the path to your GeoJSON file
geojsonFilePath = 'D:\Thesis\export.geojson';

% Read GeoJSON file and parse it
geojsonStr = fileread(geojsonFilePath);
geojsonData = jsondecode(geojsonStr);

numBuildings = numel(geojsonData.features);

% Create a cell array to store x and y coordinates for vertices of each building
xCoords = cell(numBuildings, 1);
yCoords = cell(numBuildings, 1);


for i = 1:numBuildings

    typeVariable = geojsonData.features(i).geometry.type;

    if strcmp(typeVariable, 'Polygon')
        x = geojsonData.features(i).geometry.coordinates(:, :, 1);
        y = geojsonData.features(i).geometry.coordinates(:, :, 2);
        xCoords{i} = x;
        yCoords{i} = y;

    else
        continue;
    end
end


gridWidth = round(width_in_meters);
gridHeight = round(height_in_meters);

[xCoordinate, yCoordinate] = convertlongitude2xy(xCoords, yCoords, gridWidth, gridHeight);


%% Divide the main map into 200*200 Maps and store the buildings locations in each map

gridSize = 200;

gridWidth = gridWidth * 0.5;  
gridHeight = gridHeight * 0.5;

numRows = floor(gridHeight / gridSize);
numCols = floor(gridWidth / gridSize);


allBuildingsX = cell(numRows,numCols);
allBuildingsY = cell(numRows,numCols);


for startX = 1:gridSize:gridWidth-158 
    for startY = 1:gridSize:gridHeight-137

        endX = startX + gridSize - 1;
        endY = startY + gridSize - 1;

        verticesInRegionX = {};
        verticesInRegionY = {};
        adjustedXvertices = {};
        adjustedYvertices = {};

        for i = 1:numel(xCoordinate)
            xPoly = xCoordinate{i};
            yPoly = yCoordinate{i};

            withinSection = xPoly >= startX & xPoly <= endX & ...
                yPoly >= startY & yPoly <= endY;

            if any(withinSection)
                verticesInRegionX{end+1} = xPoly(withinSection);
                verticesInRegionY{end+1} = yPoly(withinSection);
            end
        end

            [Xmap, Ymap] = meshgrid(startX:startX + 199, startY:startY + 199);


        for k = 1:numel(verticesInRegionX)
            xBuilding = verticesInRegionX{k};
            yBuilding = verticesInRegionY{k};

            adjustedXv = xBuilding - startX + 1; % convert the coordinates to a 200*200 grid
            adjustedYv = yBuilding - startY + 1;

            adjustedXvertices{end+1} = adjustedXv;
            adjustedYvertices{end+1} = adjustedYv;
        end

        allBuildingsX{(startY-1)/gridSize + 1, (startX-1)/gridSize + 1} = [allBuildingsX{(startY-1)/gridSize + 1, (startX-1)/gridSize + 1}, adjustedXvertices];
        allBuildingsY{(startY-1)/gridSize + 1, (startX-1)/gridSize + 1} = [allBuildingsY{(startY-1)/gridSize + 1, (startX-1)/gridSize + 1}, adjustedYvertices];

    end
end



%% Create 200*200*20 Occupancy Map and assign the occupied points by buildings as 1

TurinMaps = cell(size(allBuildingsX));

for i = 1:size(allBuildingsX, 1)
    for j = 1:size(allBuildingsX, 2)
        xCoordsCell = allBuildingsX{i, j};
        yCoordsCell = allBuildingsY{i, j};

        occupancyGrid = zeros(gridSize, gridSize, 20);

        for k = 1:numel(xCoordsCell)
            xbuilding = xCoordsCell{k};
            ybuilding = yCoordsCell{k};

            [xbuilding, ybuilding] = poly2cw(xbuilding, ybuilding);

            buildingHeight = randi([6, 9]); % Random integer between 6 and 9 (12-18 meter)
            mask = poly2mask(xbuilding, ybuilding, gridSize, gridSize);

            for row = 1:size(mask, 1)
                for col = 1:size(mask, 2)
                    if mask(row, col)
                        occupancyGrid(row, col, 1:buildingHeight) = 1;
                    end
                end
            end
        end

        TurinMaps{i, j} = occupancyGrid;
    end
end





%% Extract population density data from csv file

allData = {};

ds = datastore('D:\Thesis\ita_general_2020_csv\ita_general_2020.csv');

ds.SelectedVariableNames = {'longitude', 'latitude', 'ita_general_2020'};  % Specify columns of data

while hasdata(ds)
    partialData = read(ds);

    allData{end+1} = partialData;
end


%% combine all the data from the previous step to get 1 matrix

allArrays = cell(1, numel(allData));

for i = 1:numel(allData)
    tableSize = size(allData{i});
    tableArray = table2array(allData{i});

    reshapedArray = reshape(tableArray, tableSize(1), []);
    allArrays{i} = reshapedArray;
end


AllPopulationDensity = vertcat(allArrays{:});

%% find the coordinates of cells that fall in the region that we are interested (turin)

TargetLocation = AllPopulationDensity(AllPopulationDensity(:, 1) >= minLongitude & AllPopulationDensity(:, 1) <= maxLongitude & ...
    AllPopulationDensity(:, 2) >= minLatitude & AllPopulationDensity(:, 2) <= maxLatitude, :);



%% find the verteces of the cells and get the density data for each vertex and store it in a matrix


cellVertices = zeros(size(TargetLocation,1),8);
cellLengthAtEquator = 30.87; % Length of the cell at the equator (30.87 meters)
changeInLatitude = cellLengthAtEquator / 111319.9;


for i = 1:size(TargetLocation,1)

    centerLongitude = TargetLocation(i,1);
    centerLatitude = TargetLocation(i,2);

    latitudeRadians = deg2rad(centerLatitude);
    scalingFactor = cos(latitudeRadians);
    changeInLongitude = cellLengthAtEquator / (scalingFactor * 111319.9); 

    cellVertices(i,1:4) = [centerLongitude + changeInLongitude, centerLongitude + changeInLongitude, centerLongitude - changeInLongitude, centerLongitude - changeInLongitude];
    cellVertices(i,5:8) = [centerLatitude + changeInLatitude, centerLatitude - changeInLatitude, centerLatitude - changeInLatitude, centerLatitude + changeInLatitude];

end


densityValues = zeros(size(TargetLocation,1) * 4 ,3);


for i = 1:size(cellVertices, 1)

    cellLongitudes = cellVertices(i, 1:4);
    cellLatitudes = cellVertices(i, 5:8);

    cellDensity = TargetLocation(i,3);

    startIdx = (i - 1) * 4 + 1;
    endIdx = i * 4;

    densityValues(startIdx:endIdx, 1) = cellLongitudes.';
    densityValues(startIdx:endIdx, 2) = cellLatitudes.';
    densityValues(startIdx:endIdx, 3) = cellDensity;
end




%% convert latitude/longitude to xy coordinates for a desired grid size


gridWidth = round(width_in_meters);
gridHeight = round(height_in_meters);

ScalingFactor = 0.5;  

lonScale = (gridWidth / (maxLongitude - minLongitude)) * ScalingFactor;
latScale = (gridHeight / (maxLatitude - minLatitude)) * ScalingFactor;

gridCoordinatesDensity = [lonScale * (densityValues(:, 1) - minLongitude), latScale * (densityValues(:, 2) - minLatitude), densityValues(:,3)];


height = size(TurinMaps,1) * gridSize; 
width = size(TurinMaps,2) * gridSize;


gridCoordinatesDensity(gridCoordinatesDensity(:, 2) < 0, 2) = 1;
gridCoordinatesDensity(gridCoordinatesDensity(:, 1) < 0, 1) = 1;

gridCoordinatesDensity(gridCoordinatesDensity(:, 2) > height, 2) = height;
gridCoordinatesDensity(gridCoordinatesDensity(:, 1) > width, 1) = width;



%% to find the grid points which fall inside a density cell and assign the same density value to these points

numSquares = size(gridCoordinatesDensity, 1) / 4;

pointsInEachCell = {};
points_within_cell = [];

maxPointsPerBatch = 1e6;

totalPoints = 0;

for i = 1:numSquares

    startIdx = (i - 1) * 4 + 1; 
    endIdx = i * 4;

    xPoly = gridCoordinatesDensity(startIdx:endIdx, 1); 
    yPoly = gridCoordinatesDensity(startIdx:endIdx, 2);

    points_within_rect = []; 

    if any(isequal(unique(xPoly), xPoly(1)) | isequal(unique(yPoly), yPoly(1)))
        continue
    end

    x_min = min(xPoly(:));
    x_max = max(xPoly(:));
    y_min = min(yPoly(:));
    y_max = max(yPoly(:));


    [X, Y] = meshgrid(round(x_min):round(x_max), round(y_min):round(y_max));
    mesh_points = [X(:), Y(:)];

    insideCell = all(mesh_points >= min([xPoly, yPoly]), 2) & all(mesh_points <= max([xPoly, yPoly]), 2); 
    points_within_rect = mesh_points(insideCell, :); 


    if totalPoints + size(points_within_rect, 1) > maxPointsPerBatch
        pointsInEachCell{end+1} = points_within_cell;
        points_within_cell = [];
        totalPoints = 0;
    end

    points_within_cell = [points_within_cell; points_within_rect, gridCoordinatesDensity(startIdx, 3) * ones(size(points_within_rect, 1), 1)];

    totalPoints = totalPoints + size(points_within_rect, 1);
end


if ~isempty(points_within_cell)
    pointsInEachCell{end+1} = points_within_cell;
end

allDensities = vertcat(pointsInEachCell{:});


%% to divide the overal density grid into 200*200 Density Maps with normalized density


minDensity = min(allDensities(:,3));
maxDensity = max(allDensities(:,3));

onlyNormalizedDensity = (allDensities(:,3) - minDensity) / (maxDensity - minDensity);

x_range = 1:gridSize:width;
y_range = 1:gridSize:height;


numRows = floor(height / gridSize);
numCols = floor(width / gridSize);

DensityMaps = cell(numRows, numCols); 


for i = 1:numRows
    for j = 1:numCols
        x_min = x_range(j);
        x_max = x_min + gridSize - 1;
        y_min = y_range(i);
        y_max = y_min + gridSize - 1;


        grids = zeros(gridSize, gridSize);

        pointsInGrid = allDensities(allDensities(:,1) >= x_min & allDensities(:,1) <= x_max & ...
                                    allDensities(:,2) >= y_min & allDensities(:,2) <= y_max, :);

        rowNumbers = find(allDensities(:,1) >= x_min & allDensities(:,1) <= x_max & ...
                                    allDensities(:,2) >= y_min & allDensities(:,2) <= y_max);
      
        densities = onlyNormalizedDensity(rowNumbers); 

        adjustedX = round(pointsInGrid(:,1) - x_min + 1);
        adjustedY = round(pointsInGrid(:,2) - y_min + 1);

        for z = 1:length(adjustedX)
            if adjustedY(z) > 0 && adjustedX(z) > 0
                currentDensity = grids(adjustedY(z), adjustedX(z));
                newDensity = densities(z);
                if newDensity > currentDensity
                    grids(adjustedY(z), adjustedX(z)) = newDensity;
                end
            end
        end

        DensityMaps{i,j} = grids;
    end
end





%%



function [x,y] = convertlongitude2xy(lonCell, latCell, gridWidth, gridHeight)


allCoordinates = cell2mat(cellfun(@(lat, lon) [lon(:), lat(:)], latCell, lonCell , 'UniformOutput', false));

minLat = min(allCoordinates(:, 2));
minLon = min(allCoordinates(:, 1));

ScalingFactor = 0.5;

latScale = (gridHeight / (max(allCoordinates(:, 2)) - minLat)) * ScalingFactor;
lonScale = (gridWidth / (max(allCoordinates(:, 1)) - minLon)) * ScalingFactor;

gridCoordinates = [lonScale * (allCoordinates(:, 1) - minLon), latScale * (allCoordinates(:, 2) - minLat)];


figure;

currentVertex = 1;
for i = 1:length(latCell)
    numVertices = numel(latCell{i});

    plot(gridCoordinates(currentVertex:currentVertex + numVertices - 1, 1), ...
        gridCoordinates(currentVertex:currentVertex + numVertices - 1, 2), 'b-');
    hold on;

    x{i,1} = gridCoordinates(currentVertex:currentVertex + numVertices - 1, 1);  % store the coordinates in the new grid
    y{i,1} = gridCoordinates(currentVertex:currentVertex + numVertices - 1, 2);

    currentVertex = currentVertex + numVertices;
end

xlim([0 gridWidth * ScalingFactor]);
ylim([0 gridHeight * ScalingFactor]);

xlabel('Grid Longitudes');
ylabel('Grid Latitudes');

grid on;
axis equal;
end
